from django.db import models
from django.contrib import admin
from django.urls import reverse
from django.utils.translation import gettext_lazy as _
from django.utils.html import mark_safe
from django.template import engines
from tinymce.models import HTMLField
from app.common import get_request


class Page(models.Model):

    class STATUS(models.IntegerChoices):
        PUBLISHED = (0, _('Published'))
        UNPUBLISHED = (1, _('Unpublished'))
        DRAFT = (2, _('Draft'))

    title = models.CharField(max_length=256)
    slug = models.SlugField(max_length=256, unique=True, help_text=_(
        f'e.g: if the slug is "privacy-policy" then the page URL would look like "https://domain.com/privacy-policy"'))
    content = HTMLField()

    status = models.IntegerField(choices=STATUS.choices, default=STATUS.DRAFT)

    keywords = models.CharField(max_length=512, blank=True, null=True, help_text=_(
        'Keywords for SEO, e.g: keyword1, keyword2, keyword3...'))
    description = models.TextField(
        blank=True, null=True, help_text=_('A small description for SEO.'))
    head_code = models.TextField(blank=True, null=True, help_text=_(
        'A code that will be injected in "head" tag, e.g: Facebook Pixel, Google Analytics code, ...etc'))

    updated = models.DateTimeField(auto_now=True)
    created = models.DateTimeField(auto_now_add=True)

    def icon(self):
        return "<i class='fa fa-copy nav-icon'></i>"

    def __str__(self):
        return self.title

    @property
    def head(self):
        keywords = self.keywords if self.keywords is not None else ""

        return mark_safe(f"""
            <!-- Page head tags -->
            <meta name='keywords' content='{keywords}'>
            <meta name='description' content='{self.description}' >
            {self.head_code}
            <!-- / Page head tags -->
        """)

    def set_placeholders(self, placeholders):
        self._placeholders = placeholders

    def get_content(self):
        if not hasattr(self, '_placeholders') or self._placeholders is None:
            return self.content

        req = get_request()

        content = engines['django'].from_string(self.content)
        return content.render(self._placeholders, req)


class Menu(models.Model):
    name = models.CharField(max_length=40)

    class Meta:
        ordering = ('pk',)

    def total_items(self):
        return self.items.count()

    def icon(self):
        return "<i class='fa fa-list nav-icon'></i>"

    def __str__(self):
        return self.name


class MenuItem(models.Model):

    class TYPE(models.IntegerChoices):
        LINK = (0, _('Link'))
        PAGE = (1, _('Page'))

    class TARGET(models.IntegerChoices):
        _SELF = (0, _('_self'))
        _BLANK = (1, _('_blank'))
        _PARENT = (2, _('_parent'))
        _TOP = (3, _('_top'))

    title = models.CharField(max_length=40)
    item_type = models.IntegerField(
        choices=TYPE.choices, default=TYPE.LINK)
    url = models.CharField(_('URL'), max_length=256, blank=True, null=True, help_text=_(
        f'Relative URL e.g: "/games" or a full URL "https://domain.com/privacy-policy".'))
    page = models.ForeignKey(
        'Page', on_delete=models.CASCADE, blank=True, null=True)
    target = models.IntegerField(choices=TARGET.choices, default=TARGET._SELF)
    css_class = models.CharField(max_length=256, blank=True, null=True)

    menu = models.ForeignKey(
        'Menu', on_delete=models.CASCADE, related_name='items')

    order = models.IntegerField(default=0, db_index=True)

    class Meta:
        ordering = ('order',)

    def __str__(self):
        return self.title
